/** @file   rect2d.h
 * @brief   Both declaration and implementation of Rect2D - class.
 * @version $Revision: 1.1.1.1 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_ENG2D_RECT2D_H
#define H_ENG2D_RECT2D_H

#include "eng2d_dll.h"
#include "vec2d.h"

namespace eng2d {


/** @class  Rect2D
 * @brief   Represents a 2d-rectangles.
 * @author  Tomi Lamminsaari
 *
 */
class Rect2D
{
public:

  ///
  /// Constants, datatype and static members
  /// ======================================

  /** Coordinates of the topleft corner */
  Vec2D topleft;
  
  /** Coordinates of the bottomright corner */
  Vec2D bottomright;



  ///
  /// Constructors, destructor and operators
  /// ======================================

	/** Constructor.
   */
	Rect2D() :
    topleft( 0,0 ),
    bottomright( 0,0 )
  {
  }
  
	/** Constructs new rectangle.
   * @param     tl                The topleft-coordinates
   * @param     br                The bottomright - coordinates
   */
  Rect2D( const Vec2D& tl, const Vec2D& br ) :
    topleft( tl ),
    bottomright( br )
  {
  }

  /** Constructs new rectangle where topleft corner is (0,0) and bottomright
   * corner is given.
   * @param     aBottomRight      The bottomright corner.
   */
  Rect2D( const Vec2D& aBottomRight ) :
    topleft( 0,0 ),
    bottomright( aBottomRight )
  {
  }

	/** Destructor
   */
	virtual ~Rect2D()
	{
  }
  

	/** Copy constructor.
   * @param     rO                Reference to another Rect2D
   */
  Rect2D( const Rect2D& rO ) :
    topleft( rO.topleft ),
    bottomright( rO.bottomright )
  {
  }
  
	/** Assignment operator
   * @param     rO                Reference to another Rect2D
   * @return    Reference to us.
   */
  Rect2D& operator = ( const Rect2D& rO )
  {
    topleft = rO.topleft;
    bottomright = rO.bottomright;
    return *this;
  }
  
  /** Equality operator
   */
  bool operator == ( const Rect2D& rRect ) const
  {
    /*
    if ( topleft == rRect.topleft && bottomright == rRect.bottomright ) {
      return true;
    }
    */
    return false;
  }
  
  /** Inequality operator
   */
  bool operator != ( const Rect2D& rRect ) const
  {
    /*
    if ( topleft == rRect.topleft && bottomright == rRect.bottomright ) {
      return false;
    }
    */
    return true;
  }



  ///
  /// Methods
  /// =======




  ///
  /// Getter methods
  /// ==============
  
  /** Tells if rectangle is valid. It is valid only when topleft-corner
   * has smaller coordinate-values than the bottomright corner.
   * @return      <code>true</code> if rectangle is valid
   */
  virtual bool isValid() const
  {
    if ( topleft.x() > bottomright.x() || topleft.y() > bottomright.y() ) {
      return false;
    }
    return true;
  }
  
  /** Returns the width of this rectangle
   * @return    Width of the rectangle
   */
  virtual float w() const
  {
    return fabs( topleft.x() - bottomright.x() );
  }
  
  /** Returns the height of this rectangle
   * @return    Height
   */
  virtual float h() const
  {
    return fabs( topleft.y() - bottomright.y() );
  }
  
  /** Tells if this Rect2D has intersection with given Rect2D. This is
   * guaranteed to work only if rectangle is valid. See the
   * @c isValid() - method for information about when rectangle is valid.
   * @param     rRect             Another rectangle
   * @return    <code>true</code> if rectangles have intersection.
   */
  virtual bool intersects( const Rect2D& rRect ) const
  {
    if ( topleft.x() > rRect.bottomright.x() ||
         topleft.y() > rRect.bottomright.y() ||
         bottomright.x() < rRect.topleft.x() ||
         bottomright.y() < rRect.topleft.x() ) {
      return false;
    }
    return true;
  }
  
  /** Tells if given point is inside this rectangle. This method is guaranteed
   * to work only if rectangle itself is valid.
   * @param     rP                The points
   * @return    <code>true</code> if point is inside
   */
  virtual bool pointInside( const Vec2D& rP ) const
  {
    if ( rP.x() >= topleft.x() && rP.x() <= bottomright.x() &&
         rP.y() >= topleft.y() && rP.y() <= bottomright.y() ) {
      return true;
    }
    return false;
  }

  /** Adjusts the given coordinate so that it stays inside this rectangle.
   * If rect is (0,0)-(10,10) then point (-2,-10) will be mapped to (0,0) and
   * point (20,8) => (10,8).
   * @param     aPoint          The point we snap inside the rectangle.
   */
  inline void snapInside( Vec2D& aPoint ) const
  {
    if ( aPoint.vx < topleft.vx ) aPoint.vx = topleft.vx;
    if ( aPoint.vy < topleft.vy ) aPoint.vy = topleft.vy;
    if ( aPoint.vx > bottomright.vx ) aPoint.vx = bottomright.vx;
    if ( aPoint.vy > bottomright.vy ) aPoint.vy = bottomright.vy;
  }
  
protected:


  ///
  /// Members
  /// =======



private:

  ///
  /// Private members
  /// ===============

};

};  // end of namespace

#endif
